$baseUrl = "https://graph.microsoft.com/beta/deviceAppManagement/"
function MakeGetRequest(
  [parameter(Mandatory = $true)] [string] $collectionPath, 
  [parameter(Mandatory = $true)] [hashtable] $authToken)
{
  $uri = "$baseUrl$collectionPath";
  $request = "GET $uri";
  try
  {
    $response = Invoke-RestMethod $uri -Method Get -Headers $authToken;
    return $response;
  }
  catch
  {
    Write-Host -ForegroundColor Red $request;
    Write-Host -ForegroundColor Red $_.Exception.Message;
    throw;
  }
}
function MakePatchRequest(
  [parameter(Mandatory = $true)] [string] $collectionPath, 
  [parameter(Mandatory = $true)] [string] $body, 
  [parameter(Mandatory = $true)] [hashtable] $authToken)
{
  MakeRequest "PATCH" $collectionPath $body $authToken;
}

function MakePostRequest(
  [parameter(Mandatory = $true)] [string] $collectionPath, 
  [parameter(Mandatory = $true)] [string] $body, 
  [parameter(Mandatory = $true)] [hashtable] $authToken)
{
  MakeRequest "POST" $collectionPath $body $authToken;
}
function MakeRequest(
  [parameter(Mandatory = $true)] [ValidateSet("GET", "PATCH", "POST")][string] $verb, 
  [parameter(Mandatory = $true)] [string] $collectionPath, 
  [parameter(Mandatory = $true)] [string] $body, 
  [parameter(Mandatory = $true)] [hashtable] $authToken)
{
  $uri = "$baseUrl$collectionPath";
  $request = "$verb $uri";

  $headers = CloneObject $authToken;
  $headers["content-length"] = $body.Length;
  $headers["content-type"] = "application/json";

  try
  {
    $response = Invoke-RestMethod $uri -Method $verb -Headers $headers -Body $body;
    return $response;
  }
  catch
  {
    Write-Host -ForegroundColor Red $request;
    Write-Host -ForegroundColor Red $_.Exception.Message;
    throw;
  }
}

function CloneObject([parameter(Mandatory = $true)] [psobject] $deepCopyObject) 
{ 
  $stream = New-Object IO.MemoryStream; 
  $formatter = New-Object Runtime.Serialization.Formatters.Binary.BinaryFormatter; 
  $formatter.Serialize($stream, $deepCopyObject); 
  $stream.Position = 0; 
  $formatter.Deserialize($stream); 
} 

# SIG # Begin signature block
# MII9SAYJKoZIhvcNAQcCoII9OTCCPTUCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBFpQKeLzvQ6dld
# 6wYcd48XiiQeaowhM+TTKHF/SBKyl6CCInowggXMMIIDtKADAgECAhBUmNLR1FsZ
# lUgTecgRwIeZMA0GCSqGSIb3DQEBDAUAMHcxCzAJBgNVBAYTAlVTMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jvc29mdCBJZGVu
# dGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAy
# MDAeFw0yMDA0MTYxODM2MTZaFw00NTA0MTYxODQ0NDBaMHcxCzAJBgNVBAYTAlVT
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jv
# c29mdCBJZGVudGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRo
# b3JpdHkgMjAyMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALORKgeD
# Bmf9np3gx8C3pOZCBH8Ppttf+9Va10Wg+3cL8IDzpm1aTXlT2KCGhFdFIMeiVPvH
# or+Kx24186IVxC9O40qFlkkN/76Z2BT2vCcH7kKbK/ULkgbk/WkTZaiRcvKYhOuD
# PQ7k13ESSCHLDe32R0m3m/nJxxe2hE//uKya13NnSYXjhr03QNAlhtTetcJtYmrV
# qXi8LW9J+eVsFBT9FMfTZRY33stuvF4pjf1imxUs1gXmuYkyM6Nix9fWUmcIxC70
# ViueC4fM7Ke0pqrrBc0ZV6U6CwQnHJFnni1iLS8evtrAIMsEGcoz+4m+mOJyoHI1
# vnnhnINv5G0Xb5DzPQCGdTiO0OBJmrvb0/gwytVXiGhNctO/bX9x2P29Da6SZEi3
# W295JrXNm5UhhNHvDzI9e1eM80UHTHzgXhgONXaLbZ7LNnSrBfjgc10yVpRnlyUK
# xjU9lJfnwUSLgP3B+PR0GeUw9gb7IVc+BhyLaxWGJ0l7gpPKWeh1R+g/OPTHU3mg
# trTiXFHvvV84wRPmeAyVWi7FQFkozA8kwOy6CXcjmTimthzax7ogttc32H83rwjj
# O3HbbnMbfZlysOSGM1l0tRYAe1BtxoYT2v3EOYI9JACaYNq6lMAFUSw0rFCZE4e7
# swWAsk0wAly4JoNdtGNz764jlU9gKL431VulAgMBAAGjVDBSMA4GA1UdDwEB/wQE
# AwIBhjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTIftJqhSobyhmYBAcnz1AQ
# T2ioojAQBgkrBgEEAYI3FQEEAwIBADANBgkqhkiG9w0BAQwFAAOCAgEAr2rd5hnn
# LZRDGU7L6VCVZKUDkQKL4jaAOxWiUsIWGbZqWl10QzD0m/9gdAmxIR6QFm3FJI9c
# Zohj9E/MffISTEAQiwGf2qnIrvKVG8+dBetJPnSgaFvlVixlHIJ+U9pW2UYXeZJF
# xBA2CFIpF8svpvJ+1Gkkih6PsHMNzBxKq7Kq7aeRYwFkIqgyuH4yKLNncy2RtNwx
# AQv3Rwqm8ddK7VZgxCwIo3tAsLx0J1KH1r6I3TeKiW5niB31yV2g/rarOoDXGpc8
# FzYiQR6sTdWD5jw4vU8w6VSp07YEwzJ2YbuwGMUrGLPAgNW3lbBeUU0i/OxYqujY
# lLSlLu2S3ucYfCFX3VVj979tzR/SpncocMfiWzpbCNJbTsgAlrPhgzavhgplXHT2
# 6ux6anSg8Evu75SjrFDyh+3XOjCDyft9V77l4/hByuVkrrOj7FjshZrM77nq81YY
# uVxzmq/FdxeDWds3GhhyVKVB0rYjdaNDmuV3fJZ5t0GNv+zcgKCf0Xd1WF81E+Al
# GmcLfc4l+gcK5GEh2NQc5QfGNpn0ltDGFf5Ozdeui53bFv0ExpK91IjmqaOqu/dk
# ODtfzAzQNb50GQOmxapMomE2gj4d8yu8l13bS3g7LfU772Aj6PXsCyM2la+YZr9T
# 03u4aUoqlmZpxJTG9F9urJh4iIAGXKKy7aIwggbRMIIEuaADAgECAhMzAAAOCfm3
# NfbDDIs0AAAAAA4JMA0GCSqGSIb3DQEBDAUAMFoxCzAJBgNVBAYTAlVTMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMTIk1pY3Jvc29mdCBJ
# RCBWZXJpZmllZCBDUyBFT0MgQ0EgMDEwHhcNMjMwNTMxMTAyNjQ0WhcNMjMwNjAz
# MTAyNjQ0WjBLMQswCQYDVQQGEwJSTzEQMA4GA1UEBxMHQ3JhaW92YTEUMBIGA1UE
# ChMLQ2FwaHlvbiBTUkwxFDASBgNVBAMTC0NhcGh5b24gU1JMMIIBojANBgkqhkiG
# 9w0BAQEFAAOCAY8AMIIBigKCAYEA3+baVyaTqkMQ/7ixpETGahVj85ntt0PX/3zT
# vD8Z8diWA7GbF4YcRzkgrt4Tm0fsuGDYaBQKPqu+b4adySd9Zz5rhSsWrIKaGHf3
# 9MBkcm7WLOzgqChZFoYoNXPEsO+IflqJwPCwPXgz1xwLcFxize5rl7ayzPlsrm/I
# Uj6DlNgm9IbiulgGzr0YnVkUaveICf9/PCJU8wSoKZ9iTeZHEIflnmodnpYXBJrn
# EQW3jehRaZB4eD2gn2OKZwZsRckT+fSJvVuSkZ5iLNeAAk+o8mH39WBBTNYQNV3u
# +WKWBC8gXlToHvUSwmXQZ5S28Y0d0CzA+Lrb0dFn9nNuc4CpA/Fqi2RT+R12GPsI
# CVVyZZYPr3Z/TCIFDyrLUrUPbGTUiaIcha7mIs2bv+miHzJPnLnWhxcLqyJZnUoj
# PQrudUY/bkSbpJaQwT+O6lfgxIq1qpGw8XAgJ24OlTYJ44y06UR9qpYqxmyF7r8b
# TTCRZgwFLgSx5oRb8vyNwT6RnCofAgMBAAGjggIdMIICGTAMBgNVHRMBAf8EAjAA
# MA4GA1UdDwEB/wQEAwIHgDBABgNVHSUBAf8ENjA0BgorBgEEAYI3YQEABggrBgEF
# BQcDAwYcKwYBBAGCN2GDoY/BNoG0pZl+goH440yB/YauXTAdBgNVHQ4EFgQUW/de
# FwbuHWnWs+brcFTLxz8/HtcwHwYDVR0jBBgwFoAUdpw2dBPRkH1hX7MC64D0mUul
# PoUwZwYDVR0fBGAwXjBcoFqgWIZWaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3Br
# aW9wcy9jcmwvTWljcm9zb2Z0JTIwSUQlMjBWZXJpZmllZCUyMENTJTIwRU9DJTIw
# Q0ElMjAwMS5jcmwwgaUGCCsGAQUFBwEBBIGYMIGVMGQGCCsGAQUFBzAChlhodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMElE
# JTIwVmVyaWZpZWQlMjBDUyUyMEVPQyUyMENBJTIwMDEuY3J0MC0GCCsGAQUFBzAB
# hiFodHRwOi8vb25lb2NzcC5taWNyb3NvZnQuY29tL29jc3AwZgYDVR0gBF8wXTBR
# BgwrBgEEAYI3TIN9AQEwQTA/BggrBgEFBQcCARYzaHR0cDovL3d3dy5taWNyb3Nv
# ZnQuY29tL3BraW9wcy9Eb2NzL1JlcG9zaXRvcnkuaHRtMAgGBmeBDAEEATANBgkq
# hkiG9w0BAQwFAAOCAgEARn8ZL6y90zQaWfxWcNkVzfEnG2LAIY4oLOygulRfYbAg
# gO15BHia8lU+fPCRUm+wNqZ55sHB+d7opQfJhOM5tQA1f+gCHp7vcauB6MS47O82
# 8fSEdML6U6U0B1eykHqSEIVVQ+TmQtCe0O0odpQtQrmDRXRrWFToU5oLeaaRRBfo
# Tt3XI+t4y3aMEmmoe/fcHDPwk3whgiwQb97RDI5zX1TCi7dcB8bc2dlmmca+cMPg
# 5bQ0a9nnsHJcl5QeEiBYA7gTkFiH1LRL3dy1QFHn3ONIio6BsyQfLut0nQLOBRwO
# uemJ2bnrtos9fomYtmyn+brLh1EvlxxhPbb77TT7ESZj7CQ17VRaC28vecyv/Ci4
# LKqfSBee6jANLZnJ1ET2V6rch7uYD2m711LpAb2LJpLJd3SnwwS3vgtdNt3ciEnK
# yLRnGdF6QrcHnHmsCdwyHbyJOoulX+qV2SOutvbBu35qCmujCHekpo26uIaAWaD8
# KuaSzqia4CCWNGG3zI0quWDkLqAKG5VpFYDMLvKuaX3rnoknk98oR8o+UPPs37gA
# gadmZ5wFg6m2JZKZ2BUip00lGxI/vsQ6Jxprj0h69ck+GsKksgEn2lQ3HSA1PTNI
# /CXKUBboV0g2B+uYCVcKPbV4s2qaYorxncrPPuoLH3p/yM4NJBDa1iJc7P+d9vcw
# ggbRMIIEuaADAgECAhMzAAAOCfm3NfbDDIs0AAAAAA4JMA0GCSqGSIb3DQEBDAUA
# MFoxCzAJBgNVBAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
# KzApBgNVBAMTIk1pY3Jvc29mdCBJRCBWZXJpZmllZCBDUyBFT0MgQ0EgMDEwHhcN
# MjMwNTMxMTAyNjQ0WhcNMjMwNjAzMTAyNjQ0WjBLMQswCQYDVQQGEwJSTzEQMA4G
# A1UEBxMHQ3JhaW92YTEUMBIGA1UEChMLQ2FwaHlvbiBTUkwxFDASBgNVBAMTC0Nh
# cGh5b24gU1JMMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA3+baVyaT
# qkMQ/7ixpETGahVj85ntt0PX/3zTvD8Z8diWA7GbF4YcRzkgrt4Tm0fsuGDYaBQK
# Pqu+b4adySd9Zz5rhSsWrIKaGHf39MBkcm7WLOzgqChZFoYoNXPEsO+IflqJwPCw
# PXgz1xwLcFxize5rl7ayzPlsrm/IUj6DlNgm9IbiulgGzr0YnVkUaveICf9/PCJU
# 8wSoKZ9iTeZHEIflnmodnpYXBJrnEQW3jehRaZB4eD2gn2OKZwZsRckT+fSJvVuS
# kZ5iLNeAAk+o8mH39WBBTNYQNV3u+WKWBC8gXlToHvUSwmXQZ5S28Y0d0CzA+Lrb
# 0dFn9nNuc4CpA/Fqi2RT+R12GPsICVVyZZYPr3Z/TCIFDyrLUrUPbGTUiaIcha7m
# Is2bv+miHzJPnLnWhxcLqyJZnUojPQrudUY/bkSbpJaQwT+O6lfgxIq1qpGw8XAg
# J24OlTYJ44y06UR9qpYqxmyF7r8bTTCRZgwFLgSx5oRb8vyNwT6RnCofAgMBAAGj
# ggIdMIICGTAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIHgDBABgNVHSUBAf8E
# NjA0BgorBgEEAYI3YQEABggrBgEFBQcDAwYcKwYBBAGCN2GDoY/BNoG0pZl+goH4
# 40yB/YauXTAdBgNVHQ4EFgQUW/deFwbuHWnWs+brcFTLxz8/HtcwHwYDVR0jBBgw
# FoAUdpw2dBPRkH1hX7MC64D0mUulPoUwZwYDVR0fBGAwXjBcoFqgWIZWaHR0cDov
# L3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSUQlMjBW
# ZXJpZmllZCUyMENTJTIwRU9DJTIwQ0ElMjAwMS5jcmwwgaUGCCsGAQUFBwEBBIGY
# MIGVMGQGCCsGAQUFBzAChlhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3Bz
# L2NlcnRzL01pY3Jvc29mdCUyMElEJTIwVmVyaWZpZWQlMjBDUyUyMEVPQyUyMENB
# JTIwMDEuY3J0MC0GCCsGAQUFBzABhiFodHRwOi8vb25lb2NzcC5taWNyb3NvZnQu
# Y29tL29jc3AwZgYDVR0gBF8wXTBRBgwrBgEEAYI3TIN9AQEwQTA/BggrBgEFBQcC
# ARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9Eb2NzL1JlcG9zaXRv
# cnkuaHRtMAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEARn8ZL6y90zQaWfxW
# cNkVzfEnG2LAIY4oLOygulRfYbAggO15BHia8lU+fPCRUm+wNqZ55sHB+d7opQfJ
# hOM5tQA1f+gCHp7vcauB6MS47O828fSEdML6U6U0B1eykHqSEIVVQ+TmQtCe0O0o
# dpQtQrmDRXRrWFToU5oLeaaRRBfoTt3XI+t4y3aMEmmoe/fcHDPwk3whgiwQb97R
# DI5zX1TCi7dcB8bc2dlmmca+cMPg5bQ0a9nnsHJcl5QeEiBYA7gTkFiH1LRL3dy1
# QFHn3ONIio6BsyQfLut0nQLOBRwOuemJ2bnrtos9fomYtmyn+brLh1EvlxxhPbb7
# 7TT7ESZj7CQ17VRaC28vecyv/Ci4LKqfSBee6jANLZnJ1ET2V6rch7uYD2m711Lp
# Ab2LJpLJd3SnwwS3vgtdNt3ciEnKyLRnGdF6QrcHnHmsCdwyHbyJOoulX+qV2SOu
# tvbBu35qCmujCHekpo26uIaAWaD8KuaSzqia4CCWNGG3zI0quWDkLqAKG5VpFYDM
# LvKuaX3rnoknk98oR8o+UPPs37gAgadmZ5wFg6m2JZKZ2BUip00lGxI/vsQ6Jxpr
# j0h69ck+GsKksgEn2lQ3HSA1PTNI/CXKUBboV0g2B+uYCVcKPbV4s2qaYorxncrP
# PuoLH3p/yM4NJBDa1iJc7P+d9vcwggdaMIIFQqADAgECAhMzAAAABkoa+s8FYWp0
# AAAAAAAGMA0GCSqGSIb3DQEBDAUAMGMxCzAJBgNVBAYTAlVTMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xNDAyBgNVBAMTK01pY3Jvc29mdCBJRCBWZXJp
# ZmllZCBDb2RlIFNpZ25pbmcgUENBIDIwMjEwHhcNMjEwNDEzMTczMTU0WhcNMjYw
# NDEzMTczMTU0WjBaMQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMSswKQYDVQQDEyJNaWNyb3NvZnQgSUQgVmVyaWZpZWQgQ1MgRU9D
# IENBIDAxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAx+PIP/Qh3cYZ
# wLvFy6uuJ4fTp3ln7Gqs7s8lTVyfgOJWP1aABwk2/oxdVjfSHUq4MTPXilL57qi/
# fH7YndEK4Knd3u5cedFwr2aHSTp6vl/PL1dAL9sfoDvNpdG0N/R84AhYNpBQThpO
# 4/BqxmCgl3iIRfhh2oFVOuiTiDVWvXBg76bcjnHnEEtXzvAWwJu0bBU7oRRqQed4
# VXJtICVt+ZoKUSjqY5wUlhAdwHh+31BnpBPCzFtKViLp6zEtRyOxRegagFU+yLgX
# vvmd07IDN0S2TLYuiZjTw+kcYOtoNgKr7k0C6E9Wf3H4jHavk2MxqFptgfL0gL+z
# bSb+VBNKiVT0mqzXJIJmWmqw0K+D3MKfmCer3e3CbrP+F5RtCb0XaE0uRcJPZJjW
# wciDBxBIbkNF4GL12hl5vydgFMmzQcNuodKyX//3lLJ1q22roHVS1cgtsLgpjWYZ
# lBlhCTcXJeZ3xuaJvXZB9rcLCX15OgXL21tUUwJCLE27V5AGZxkO3i54mgSCswtO
# mWU4AKd/B/e3KtXv6XBURKuAteez1EpgloaZwQej9l5dN9Uh8W19BZg9IlLl+xHR
# X4vDiMWAUf/7ANe4MoS98F45r76IGJ0hC02EMuMZxAErwZj0ln0aL53EzlMa5JCi
# RObb0UoLHfGSdNJsMg0uj3DAQDdVWTECAwEAAaOCAg4wggIKMA4GA1UdDwEB/wQE
# AwIBhjAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQUdpw2dBPRkH1hX7MC64D0
# mUulPoUwVAYDVR0gBE0wSzBJBgRVHSAAMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTAZBgkr
# BgEEAYI3FAIEDB4KAFMAdQBiAEMAQTASBgNVHRMBAf8ECDAGAQH/AgEAMB8GA1Ud
# IwQYMBaAFNlBKbAPD2Ns72nX9c0pnqRIajDmMHAGA1UdHwRpMGcwZaBjoGGGX2h0
# dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29mdCUyMElE
# JTIwVmVyaWZpZWQlMjBDb2RlJTIwU2lnbmluZyUyMFBDQSUyMDIwMjEuY3JsMIGu
# BggrBgEFBQcBAQSBoTCBnjBtBggrBgEFBQcwAoZhaHR0cDovL3d3dy5taWNyb3Nv
# ZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNyb3NvZnQlMjBJRCUyMFZlcmlmaWVkJTIw
# Q29kZSUyMFNpZ25pbmclMjBQQ0ElMjAyMDIxLmNydDAtBggrBgEFBQcwAYYhaHR0
# cDovL29uZW9jc3AubWljcm9zb2Z0LmNvbS9vY3NwMA0GCSqGSIb3DQEBDAUAA4IC
# AQBqLwmf2LB1QjUga0G7zFkbGd8NBQLHP0KOFBWNJFZiTtKfpO0bZ2Wfs6v5vqIK
# jE32Q6M89G4ZkVcvWuEAA+dvjLThSy89Y0//m/WTSKwYtiR1Ewn7x1kw/Fg93wQp
# s2C1WUj+00/6uNrF+d4MVJxV1HoBID+95ZIW0KkqZopnOA4w5vP4T5cBprZQAlP/
# vMGyB0H9+pHNo0jT9Q8gfKJNzHS9i1DgBmmufGdW9TByuno8GAizFMhLlIs08b5l
# ilIkE5z3FMAUAr+XgII1FNZnb43OI6Qd2zOijbjYfursXUCNHC+RSwJGm5ULzPym
# YggnJ+khJOq7oSlqPGpbr70hGBePw/J7/mmSqp7hTgt0mPikS1i4ap8x+P3yemYS
# hnFrgV1752TI+As69LfgLthkITvf7bFHB8vmIhadZCOS0vTCx3B+/OVcEMLNO2bJ
# 0O9ikc1JqR0Fvqx7nAwMRSh3FVqosgzBbWnVkQJq7oWFwMVfFIYn6LPRZMt48u6i
# MUCFBSPddsPA/6k85mEv+08U5WCQ7ydj1KVV2THre/8mLHiem9wf/CzohqRntxM2
# E/x+NHy6TBMnSPQRqhhNfuOgUDAWEYmlM/ZHGaPIb7xOvfVyLQ/7l6YfogT3eptw
# p4GOGRjH5z+gG9kpBIx8QrRl6OilnlxRExokmMflL7l12TCCB54wggWGoAMCAQIC
# EzMAAAAHh6M0o3uljhwAAAAAAAcwDQYJKoZIhvcNAQEMBQAwdzELMAkGA1UEBhMC
# VVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjFIMEYGA1UEAxM/TWlj
# cm9zb2Z0IElkZW50aXR5IFZlcmlmaWNhdGlvbiBSb290IENlcnRpZmljYXRlIEF1
# dGhvcml0eSAyMDIwMB4XDTIxMDQwMTIwMDUyMFoXDTM2MDQwMTIwMTUyMFowYzEL
# MAkGA1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjE0MDIG
# A1UEAxMrTWljcm9zb2Z0IElEIFZlcmlmaWVkIENvZGUgU2lnbmluZyBQQ0EgMjAy
# MTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALLwwK8ZiCji3VR6TEls
# aQhVCbRS/3pK+MHrJSj3Zxd3KU3rlfL3qrZilYKJNqztA9OQacr1AwoNcHbKBLbs
# QAhBnIB34zxf52bDpIO3NJlfIaTE/xrweLoQ71lzCHkD7A4As1Bs076Iu+mA6cQz
# sYYH/Cbl1icwQ6C65rU4V9NQhNUwgrx9rGQ//h890Q8JdjLLw0nV+ayQ2Fbkd242
# o9kH82RZsH3HEyqjAB5a8+Ae2nPIPc8sZU6ZE7iRrRZywRmrKDp5+TcmJX9MRff2
# 41UaOBs4NmHOyke8oU1TYrkxh+YeHgfWo5tTgkoSMoayqoDpHOLJs+qG8Tvh8Sni
# fW2Jj3+ii11TS8/FGngEaNAWrbyfNrC69oKpRQXY9bGH6jn9NEJv9weFxhTwyvx9
# OJLXmRGbAUXN1U9nf4lXezky6Uh/cgjkVd6CGUAf0K+Jw+GE/5VpIVbcNr9rNE50
# Sbmy/4RTCEGvOq3GhjITbCa4crCzTTHgYYjHs1NbOc6brH+eKpWLtr+bGecy9Crw
# Qyx7S/BfYJ+ozst7+yZtG2wR461uckFu0t+gCwLdN0A6cFtSRtR8bvxVFyWwTtgM
# MFRuBa3vmUOTnfKLsLefRaQcVTgRnzeLzdpt32cdYKp+dhr2ogc+qM6K4CBI5/j4
# VFyC4QFeUP2YAidLtvpXRRo3AgMBAAGjggI1MIICMTAOBgNVHQ8BAf8EBAMCAYYw
# EAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNlBKbAPD2Ns72nX9c0pnqRIajDm
# MFQGA1UdIARNMEswSQYEVR0gADBBMD8GCCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wGQYJKwYBBAGC
# NxQCBAweCgBTAHUAYgBDAEEwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTI
# ftJqhSobyhmYBAcnz1AQT2ioojCBhAYDVR0fBH0wezB5oHegdYZzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSWRlbnRpdHkl
# MjBWZXJpZmljYXRpb24lMjBSb290JTIwQ2VydGlmaWNhdGUlMjBBdXRob3JpdHkl
# MjAyMDIwLmNybDCBwwYIKwYBBQUHAQEEgbYwgbMwgYEGCCsGAQUFBzAChnVodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMElk
# ZW50aXR5JTIwVmVyaWZpY2F0aW9uJTIwUm9vdCUyMENlcnRpZmljYXRlJTIwQXV0
# aG9yaXR5JTIwMjAyMC5jcnQwLQYIKwYBBQUHMAGGIWh0dHA6Ly9vbmVvY3NwLm1p
# Y3Jvc29mdC5jb20vb2NzcDANBgkqhkiG9w0BAQwFAAOCAgEAfyUqnv7Uq+rdZgrb
# VyNMul5skONbhls5fccPlmIbzi+OwVdPQ4H55v7VOInnmezQEeW4LqK0wja+fBzn
# ANbXLB0KrdMCbHQpbLvG6UA/Xv2pfpVIE1CRFfNF4XKO8XYEa3oW8oVH+KZHgIQR
# IwAbyFKQ9iyj4aOWeAzwk+f9E5StNp5T8FG7/VEURIVWArbAzPt9ThVN3w1fAZkF
# 7+YU9kbq1bCR2YD+MtunSQ1Rft6XG7b4e0ejRA7mB2IoX5hNh3UEauY0byxNRG+f
# T2MCEhQl9g2i2fs6VOG19CNep7SquKaBjhWmirYyANb0RJSLWjinMLXNOAga10n8
# i9jqeprzSMU5ODmrMCJE12xS/NWShg/tuLjAsKP6SzYZ+1Ry358ZTFcx0FS/mx2v
# SoU8s8HRvy+rnXqyUJ9HBqS0DErVLjQwK8VtsBdekBmdTbQVoCgPCqr+PDPB3xaj
# Ynzevs7eidBsM71PINK2BoE2UfMwxCCX3mccFgx6UsQeRSdVVVNSyALQe6PT1241
# 8xon2iDGE81OGCreLzDcMAZnrUAx4XQLUz6ZTl65yPUiOh3k7Yww94lDf+8oG2oZ
# mDh5O1Qe38E+M3vhKwmzIeoB1dVLlz4i3IpaDcR+iuGjH2TdaC1ZOmBXiCRKJLj4
# DT2uhJ04ji+tHD6n58vhavFIrmcxghokMIIaIAIBATBxMFoxCzAJBgNVBAYTAlVT
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMTIk1pY3Jv
# c29mdCBJRCBWZXJpZmllZCBDUyBFT0MgQ0EgMDECEzMAAA4J+bc19sMMizQAAAAA
# DgkwDQYJYIZIAWUDBAIBBQCggYYwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQw
# LwYJKoZIhvcNAQkEMSIEIEMsS8XRj2Lz+8FwK+/6mRZWRzf/ArqvHuvJsZPNdyGg
# MDgGCisGAQQBgjcCAQwxKjAooCaAJABBAGQAdgBhAG4AYwBlAGQAIABJAG4AcwB0
# AGEAbABsAGUAcjANBgkqhkiG9w0BAQEFAASCAYC/MDg+hVO3Uq79cvezSfw6MF39
# Wr7T767KO0rnvZJXF+7NnSkvozZ5kjyL23kjAisZwQYUymvHT0DYKKeyfDIXMlQf
# JTKeQ/6ANoYBDlB4Vgic9XqjtRq92M/hePxWHAxbR2NppRZfdwbyeqNviJjTfBl9
# j5DYkxPTuSiqPotYrCN6kcp6NTZ0ctej+9ugWitFkYoiJaJE3M1N+aRiKBYPfvQt
# /DquG+nlVyEMbX956x32s8u79H+Nlzedf6QRDGkYVobcnK7upHO3qLij7OHPc0sG
# W06E6cpGAvAUrshFqMz3uTHBZ0pjF002+o2U1ObLpXQK6ANBy5w+dXRAF1k67HMi
# 5Thp2TYJMfd3y2FvQBQaIMHS5EgfNcHEebCgFhg3l/PRxITxgtgAjZeI/IoJSzfY
# mGw8YrhxbsIBZVlk+UEpeCHkIWfD6vve6og6i/QcDKT8YGUCKhLrDn9r5dcFU/6E
# qM5sqTpTG1NXb/RO29FNcHYCLf0dsSrmUPxPVh2hghd7MIIXdwYKKwYBBAGCNwMD
# ATGCF2cwghdjBgkqhkiG9w0BBwKgghdUMIIXUAIBAzEPMA0GCWCGSAFlAwQCAQUA
# MIIBaQYLKoZIhvcNAQkQAQSgggFYBIIBVDCCAVACAQEGCisGAQQBhFkKAwEwMTAN
# BglghkgBZQMEAgEFAAQgS/guQjBjTA+PcjNWhz4dHADLMnt4ShADtnbUOW1BSRYC
# BmRstObInhgTMjAyMzA1MzExMzU0MzMuNjM3WjAEgAIB9KCB6KSB5TCB4jELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEtMCsGA1UECxMkTWljcm9z
# b2Z0IElyZWxhbmQgT3BlcmF0aW9ucyBMaW1pdGVkMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjo5MUEyLTk2NkMtNjNGQjE1MDMGA1UEAxMsTWljcm9zb2Z0IFB1Ymxp
# YyBSU0EgVGltZSBTdGFtcGluZyBBdXRob3JpdHmgghHzMIIHgjCCBWqgAwIBAgIT
# MwAAAAXlzw//Zi7JhwAAAAAABTANBgkqhkiG9w0BAQwFADB3MQswCQYDVQQGEwJV
# UzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMUgwRgYDVQQDEz9NaWNy
# b3NvZnQgSWRlbnRpdHkgVmVyaWZpY2F0aW9uIFJvb3QgQ2VydGlmaWNhdGUgQXV0
# aG9yaXR5IDIwMjAwHhcNMjAxMTE5MjAzMjMxWhcNMzUxMTE5MjA0MjMxWjBhMQsw
# CQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYD
# VQQDEylNaWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMDCC
# AiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAJ5851Jj/eDFnwV9Y7UGIqMc
# HtfnlzPREwW9ZUZHd5HBXXBvf7KrQ5cMSqFSHGqg2/qJhYqOQxwuEQXG8kB41wsD
# JP5d0zmLYKAY8Zxv3lYkuLDsfMuIEqvGYOPURAH+Ybl4SJEESnt0MbPEoKdNihwM
# 5xGv0rGofJ1qOYSTNcc55EbBT7uq3wx3mXhtVmtcCEr5ZKTkKKE1CxZvNPWdGWJU
# PC6e4uRfWHIhZcgCsJ+sozf5EeH5KrlFnxpjKKTavwfFP6XaGZGWUG8TZaiTogRo
# AlqcevbiqioUz1Yt4FRK53P6ovnUfANjIgM9JDdJ4e0qiDRm5sOTiEQtBLGd9Vhd
# 1MadxoGcHrRCsS5rO9yhv2fjJHrmlQ0EIXmp4DhDBieKUGR+eZ4CNE3ctW4uvSDQ
# VeSp9h1SaPV8UWEfyTxgGjOsRpeexIveR1MPTVf7gt8hY64XNPO6iyUGsEgt8c2P
# xF87E+CO7A28TpjNq5eLiiunhKbq0XbjkNoU5JhtYUrlmAbpxRjb9tSreDdtACpm
# 3rkpxp7AQndnI0Shu/fk1/rE3oWsDqMX3jjv40e8KN5YsJBnczyWB4JyeeFMW3JB
# fdeAKhzohFe8U5w9WuvcP1E8cIxLoKSDzCCBOu0hWdjzKNu8Y5SwB1lt5dQhABYy
# zR3dxEO/T1K/BVF3rV69AgMBAAGjggIbMIICFzAOBgNVHQ8BAf8EBAMCAYYwEAYJ
# KwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFGtpKDo1L0hjQM972K9J6T7ZPdshMFQG
# A1UdIARNMEswSQYEVR0gADBBMD8GCCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYI
# KwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwDwYDVR0TAQH/BAUw
# AwEB/zAfBgNVHSMEGDAWgBTIftJqhSobyhmYBAcnz1AQT2ioojCBhAYDVR0fBH0w
# ezB5oHegdYZzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWlj
# cm9zb2Z0JTIwSWRlbnRpdHklMjBWZXJpZmljYXRpb24lMjBSb290JTIwQ2VydGlm
# aWNhdGUlMjBBdXRob3JpdHklMjAyMDIwLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQw
# gYEGCCsGAQUFBzAChnVodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2Nl
# cnRzL01pY3Jvc29mdCUyMElkZW50aXR5JTIwVmVyaWZpY2F0aW9uJTIwUm9vdCUy
# MENlcnRpZmljYXRlJTIwQXV0aG9yaXR5JTIwMjAyMC5jcnQwDQYJKoZIhvcNAQEM
# BQADggIBAF+Idsd+bbVaFXXnTHho+k7h2ESZJRWluLE0Oa/pO+4ge/XEizXvhs0Y
# 7+KVYyb4nHlugBesnFqBGEdC2IWmtKMyS1OWIviwpnK3aL5JedwzbeBF7POyg6IG
# G/XhhJ3UqWeWTO+Czb1c2NP5zyEh89F72u9UIw+IfvM9lzDmc2O2END7MPnrcjWd
# QnrLn1Ntday7JSyrDvBdmgbNnCKNZPmhzoa8PccOiQljjTW6GePe5sGFuRHzdFt8
# y+bN2neF7Zu8hTO1I64XNGqst8S+w+RUdie8fXC1jKu3m9KGIqF4aldrYBamyh3g
# 4nJPj/LR2CBaLyD+2BuGZCVmoNR/dSpRCxlot0i79dKOChmoONqbMI8m04uLaEHA
# v4qwKHQ1vBzbV/nG89LDKbRSSvijmwJwxRxLLpMQ/u4xXxFfR4f/gksSkbJp7oqL
# wliDm/h+w0aJ/U5ccnYhYb7vPKNMN+SZDWycU5ODIRfyoGl59BsXR/HpRGtiJquO
# YGmvA/pk5vC1lcnbeMrcWD/26ozePQ/TWfNXKBOmkFpvPE8CH+EeGGWzqTCjdAsn
# o2jzTeNSxlx3glDGJgcdz5D/AAxw9Sdgq/+rY7jjgs7X6fqPTXPmaCAJKVHAP19o
# EjJIBwD1LyHbaEgBxFCogYSOiUIr0Xqcr1nJfiWG2GwYe6ZoAF1bMIIHnjCCBYag
# AwIBAgITMwAAACShlA+BLdKQowAAAAAAJDANBgkqhkiG9w0BAQwFADBhMQswCQYD
# VQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQD
# EylNaWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMDAeFw0y
# MjA3MjgxODE2NDRaFw0yMzA3MjgxODE2NDRaMIHiMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOjkxQTIt
# OTY2Qy02M0ZCMTUwMwYDVQQDEyxNaWNyb3NvZnQgUHVibGljIFJTQSBUaW1lIFN0
# YW1waW5nIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIB
# AM1Izmm8A9POZNhFywjRi1eOfDgjh5L2tG2B+IbTLdpsE5tvjdBKpAMb7xFzZFpa
# UY8TCkhdP7EzoFkqVeU+FfHxEirzidR6HMtaq1sC6LjcH6STfr94lT7e8LppavyI
# YR11uqKOL86Zh9twSOnoCgKUUyR4P/q1alEkv0dIj6JVSgooOHiHsec4brMN/Nvz
# FU4N4g7eBT/JN5E246gRpVrZ2OjT1dhEydsGnr1JF4m5D5kfAv+FmatVNO6qII81
# NDKMBsCbyN8v+0gl9WqSsDQ29PAcn8J1WJFwVrG693rENxghOgAvmm48W4pDK4Mw
# hMH1c6pkzJYvvBMD4jPGy7R+Dru6Mmc3VG2DsScDsZSE0khsAyXD4x5pTcKaJLLe
# yg7LWNfWnGfgrK8LlCW14jsn+vPk2hSRWa3SyBpKVr4GwmNW1HLuTQGBgFDS9DwX
# 4A9QA25w25x7Po5+R1B1Lk4UsOJXpeHwiUiKH51AxEkkz0u3Iv1ch4CJAeWhf3+X
# Eb3JFkmCnF+mBvmx4SqrCSbhvxKiQ1tACv2EJxloIO1W0+1fLNL7MYDvZ3gWT0jt
# RcKGznRuHvRtcBWFaAA0gpMiSI8VAo1ZjA08EZfAc4PglRgzzIUjXUdoRmtynJ0X
# PfVmUy/XY/YE42mwg+A549McNmMfZxrjIfRKcmVz3Ie1AgMBAAGjggHLMIIBxzAd
# BgNVHQ4EFgQU+awLvntg0P02HVnlLwqnyOnqtmowHwYDVR0jBBgwFoAUa2koOjUv
# SGNAz3vYr0npPtk92yEwbAYDVR0fBGUwYzBhoF+gXYZbaHR0cDovL3d3dy5taWNy
# b3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwUHVibGljJTIwUlNBJTIw
# VGltZXN0YW1waW5nJTIwQ0ElMjAyMDIwLmNybDB5BggrBgEFBQcBAQRtMGswaQYI
# KwYBBQUHMAKGXWh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY2VydHMv
# TWljcm9zb2Z0JTIwUHVibGljJTIwUlNBJTIwVGltZXN0YW1waW5nJTIwQ0ElMjAy
# MDIwLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMA4G
# A1UdDwEB/wQEAwIHgDBmBgNVHSAEXzBdMFEGDCsGAQQBgjdMg30BATBBMD8GCCsG
# AQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVw
# b3NpdG9yeS5odG0wCAYGZ4EMAQQCMA0GCSqGSIb3DQEBDAUAA4ICAQAbTbwjfY0E
# n7i+nbrNUBMz72lIYLCseiWGIASLkHPgbotRlcqXAT5cQt4XN84fiyQjn1wU9x9X
# pXfhpYsSc9b1a33WoCV5DeLCwr8v7+N1u33yLE98CHemZrb66fSbm919vpcYp3Oy
# N3vvs138YQobvjkChk19LITIjbaq3T41fydA3HTUVnLX7m/nr1Uv/L/2/v1lHI9p
# 52yFaI0Ssj5GyrKbhqMdtS0opPXjIeln4VfXDBx1inlbNwozqPfeesAkum4yOVSO
# pxp+WAYDls/YdiIqfDkJqbjcuRRxP2pK9x/rb4O+JruQzFrQQQbLuxXBQTkZokvq
# AqnKX9IugtG2M+k21Ke3pe+ve/+LXclooh1wf4X/sDhNybDW6Jpa+eq4jytYXB9Y
# skDto8E3cAWQNXQ7ZuZ5zWXvE6I30r+mZ4gIyd4CJX1EjRMDgN3M5g3UqxPQ9/Nc
# tyRAw9uHG+ggKYBXvFCMAZp1kf5Cv7JLwPOPC9epFFBaciik3pHLnhbQV10b6fCh
# UPmCWXRdnnFN9O0U1uX1hvCR/8CNBeJP8htMlnBTRngbLS+69Jk6H+5/GZnQprdn
# ILBEkfhEXT+3EIZmrryLay42zFBETS8MB57WewOiaQ6zxMe6vc1tFNyfjzSlP668
# hQ/hIRNybC0hzC0dCNReBKGxlgLjnygKkqGCAscwggIwAgEBMIIBEKGB6KSB5TCB
# 4jELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
# ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEtMCsGA1UECxMk
# TWljcm9zb2Z0IElyZWxhbmQgT3BlcmF0aW9ucyBMaW1pdGVkMSYwJAYDVQQLEx1U
# aGFsZXMgVFNTIEVTTjo5MUEyLTk2NkMtNjNGQjE1MDMGA1UEAxMsTWljcm9zb2Z0
# IFB1YmxpYyBSU0EgVGltZSBTdGFtcGluZyBBdXRob3JpdHmiIwoBATAHBgUrDgMC
# GgMVAEFw5SzAq/hNT/MjzS/PjIEodkY3oGcwZaRjMGExCzAJBgNVBAYTAlVTMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29m
# dCBQdWJsaWMgUlNBIFRpbWVzdGFtcGluZyBDQSAyMDIwMA0GCSqGSIb3DQEBBQUA
# AgUA6CG/ZzAiGA8yMDIzMDUzMTEyNDMxOVoYDzIwMjMwNjAxMTI0MzE5WjB0MDoG
# CisGAQQBhFkKBAExLDAqMAoCBQDoIb9nAgEAMAcCAQACAh2eMAcCAQACAhFFMAoC
# BQDoIxDnAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAIAgEA
# AgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQEFBQADgYEAZuvZB4JiumTYHBkt
# ROrKLma7M5G7JYIYvvnlsp8ZGRXEjOcr4N7o4fhHDB+3p1Kni5ADvNjKVLkd9grQ
# V/Q9/jB9JkLat+QCEhLxavgH//hEqyxYTMu4q+A7DcwbQg1yxDhm92Tp65kS2dtT
# Xw2+M9bEgdL4h0OiBWIBn7VRphAxggPUMIID0AIBATB4MGExCzAJBgNVBAYTAlVT
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jv
# c29mdCBQdWJsaWMgUlNBIFRpbWVzdGFtcGluZyBDQSAyMDIwAhMzAAAAJKGUD4Et
# 0pCjAAAAAAAkMA0GCWCGSAFlAwQCAQUAoIIBLTAaBgkqhkiG9w0BCQMxDQYLKoZI
# hvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEIE8x4kphANnbcgdsTPauuNxIwTp60khU
# qefYn+GGs+H2MIHdBgsqhkiG9w0BCRACLzGBzTCByjCBxzCBoAQg0jFVm0Ghvjvu
# cCH72W8fZ8imIEtyopmAfw+pb1Prjn8wfDBlpGMwYTELMAkGA1UEBhMCVVMxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0
# IFB1YmxpYyBSU0EgVGltZXN0YW1waW5nIENBIDIwMjACEzMAAAAkoZQPgS3SkKMA
# AAAAACQwIgQgjXEfN8GLm+x08jilJ4vCq0ErSAQrUpTp2jul9HuN1XowDQYJKoZI
# hvcNAQELBQAEggIAE/7fS0y8VezmfXu3h8wiuNKR69s3TdVuPSJnTjOj8Agkypeb
# oroGfeD2wE3Qw6dr9pr6C4tWNR2wsJBA+WRmoVCCeCap0VCLo42Ux/COEmeqIyGA
# dDuhnBNRSaCXIgFWTHyAkwU6C5PudkRrPUuKvDhwA33DvhsXx6O4tw3j59XLIF/A
# IWohF1bC50v7jVbT2D9/Qrt41F366UrezS0SBExRy3oyn/8HoiwPioD5WXPiHZVR
# EDJzfXNGEQajL0qfN7zQue0BB08IBZseSEj5ae3fJhy07QRgf1QEUQRdNqGJkisP
# bTDTuuJKhe1lg0s9nfAhWirEMOaCBMTikYi7eiebXHh3xg/6vjfckzaHUzplNX2M
# G2ilEcdJD0bL2FaTxu1lXr3lxkIG3XUEmZHVs4HGk9tbHk8uWwCOmIPCCNBnBoYG
# dQ8siUlJsnvVVYWg61YmjOmVHdTOfwOKxk33NZVkWl9bDAFI9BCJChtF0i7uUBBQ
# EGxLmJsYKg8QtPauKmzH4Q1CX+BLzZ+W/3uzqlGQPH28pp1Rdo8rVHeNo9ils0B7
# +zo/MGwHqIjORx95HbxDpjcVDicCu546dBW04V1/xmHz3QRxm6XNvePHxlmeRmzy
# iwrPoXKEHUVeuA7xOH1G8/wSDqwy+zoKZBKlYfBbKnIUQm5Tqm1mqjjEVbo=
# SIG # End signature block
